/******************************************************************************
 * Public header file for the UART/SCI driver.
 * 
 * Freescale Semiconductor Inc.
 * (c) Copyright 2004-2011 Freescale Semiconductor, Inc.
 * ALL RIGHTS RESERVED.
 *
*******************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
***************************************************************************//*!
******************************************************************************/

/****************************************************************************** 
 * On the receive side, this module keeps a small circular buffer, and calls
 * a callback function whenever a byte is received. The application can
 * retrieve bytes from the buffer at it's convenience, as long as it does so
 * before the driver's buffer fills up.
 *
 * On the transmit side, this module keeps a list of buffers to be
 * transmitted, and calls an application callback function when the entire
 * buffer has been sent. The application is responsible for ensuring that the
 * buffer is available until the callback function is called.
 *
 * If both SCI ports are enabled, each has it's own Rx circular buffer and
 * list of Tx buffers.
 *****************************************************************************/

#ifndef __UART_INTERFACE_H__
#define __UART_INTERFACE_H__

/******************************************************************************
*******************************************************************************
* Includes
*******************************************************************************
******************************************************************************/

#include "EmbeddedTypes.h"
#include "PortConfig.h"

/******************************************************************************
*******************************************************************************
* Public macros
*******************************************************************************
******************************************************************************/

/******************************************************************************
 * Name: gUart0_Enabled_d
 * Description: Specifies if the UART_1 HW interface is used
 *****************************************************************************/  
#ifndef gUart0_Enabled_d
#define gUart0_Enabled_d    TRUE //FALSE //
#endif
 
/******************************************************************************
 * Name: gUart1_Enabled_d
 * Description: Specifies if the UART_2 HW interface is used
 *****************************************************************************/  
#ifndef gUart1_Enabled_d
#define gUart1_Enabled_d   FALSE //TRUE //
#endif 

/******************************************************************************
 * Name: gUart0_c
 * Description: hardware UART module associated to the first UART peripheral
 *****************************************************************************/ 
#ifndef gUart0_c
#define  gUart0_c                gUART_HW_MOD_0_c  
#endif

/******************************************************************************
 * Name: gUart1_c
 * Description: hardware UART module associated to the first UART peripheral
 *****************************************************************************/
#ifndef gUart1_c
#define  gUart1_c                gUART_HW_MOD_1_c
#endif



/******************************************************************************
 * Name: gUART_TX_THRESHOLD_c
 * Description: UART TX threshold value 
 *****************************************************************************/
#ifndef gUART_TX_THRESHOLD_c 
#define gUART_TX_THRESHOLD_c    4
#endif

/******************************************************************************
 * Name: gUART_RX_THRESHOLD_c
 * Description: UART RX threshold value 
 *****************************************************************************/
#ifndef gUART_RX_THRESHOLD_c 
#define gUART_RX_THRESHOLD_c    1
#endif

/******************************************************************************
 * Name: gUart_PortDefault_d
 * Description: specifies the default used port
 *****************************************************************************/
#ifndef gUart_PortDefault_d
#if (gUart0_Enabled_d == TRUE)
#define gUart_PortDefault_d     0
#elif (gUart1_Enabled_d == TRUE)
#define gUart_PortDefault_d     1
#endif 
#endif


/******************************************************************************
 * Name: gUart_TransmitBuffers_c
 * Description: Number of entries in the transmit-buffers-in-waiting list
 *****************************************************************************/
#ifndef gUart_TransmitBuffers_c
#define gUart_TransmitBuffers_c         3
#endif /* gUart_TransmitBuffers_c */

/******************************************************************************
 * Name: gUart0_ReceiveBufferSize_c
 * Description: Size of the UART_1 driver Rx circular buffer. These buffers are used to
 *              hold received bytes until the application can retrieve them via the
 *              UartX_GetBytesFromRxBuffer() functions, and are not otherwise accessible
 *              from outside the driver. The size DOES NOT need to be a power of two.
 *****************************************************************************/
#ifndef gUart0_ReceiveBufferSize_c
#define gUart0_ReceiveBufferSize_c      250
#endif /* gUart0_ReceiveBufferSize_c */

/******************************************************************************
 * Name: gUart1_ReceiveBufferSize_c
 * Description: Size of the UART_2 driver Rx circular buffer. These buffers are used to
 *              hold received bytes until the application can retrieve them via the
 *              UartX_GetBytesFromRxBuffer() functions, and are not otherwise accessible
 *              from outside the driver. The size DOES NOT need to be a power of two.
 *****************************************************************************/
#ifndef gUart1_ReceiveBufferSize_c
#define gUart1_ReceiveBufferSize_c      250
#endif /* gUart1_ReceiveBufferSize_c */

\
/****************************************************************************** 
 * Name: Baud Rates
 * Description: UART standard baud rates definitions
 *****************************************************************************/

#define gUARTBaudRate1200_c	        1200UL
#define gUARTBaudRate2400_c 	        2400UL
#define gUARTBaudRate4800_c 	        4800UL
#define gUARTBaudRate9600_c 	        9600UL
#define gUARTBaudRate19200_c 	        19200UL
#define gUARTBaudRate38400_c 	        38400UL
#define gUARTBaudRate57600_c 	        57600UL
#define gUARTBaudRate115200_c 	        115200UL
#define gUARTBaudRate230400_c           230400UL

/****************************************************************************** 
 * UART_0 configuration 
 * 
 * gUART0_Parity_c - must be one of the following: gUartParityNone_c, gUartParityEven_c, gUartParityOdd_c
 * gUART0_DataBits_c - must be one of the following: gUart_8bits_c, gUart_9bits_c
 * gUART0_Baudrate_c - any standard or non standard baudrate (bps)

 *****************************************************************************/
#if gUart0_Enabled_d

#ifndef gUART0_Parity_c
#define gUART0_Parity_c                     gUartParityNone_c
#endif

#ifndef gUART0_DataBits_c
#define gUART0_DataBits_c                   gUart_8bits_c
#endif

#ifndef gUART0_Baudrate_c
//#define gUART0_Baudrate_c                   gUARTBaudRate115200_c     
#define gUART0_Baudrate_c                   gUARTBaudRate38400_c                //Baudate (Radio Utility GUI currently set to 38400)
#endif


#endif


/******************************************************************************
 * Uart1 configuration 
 * 
 * gUART1_Parity_c - must be one of the following: gUartParityNone_c, gUartParityEven_c, gUartParityOdd_c
 * gUART1_DataBits_c - must be one of the following: gUart_8bits_c, gUart_9bits_c
 * gUART1_Baudrate_c - any standard or non standard baudrate (bps)

 *****************************************************************************/

#if gUart1_Enabled_d

#ifndef gUART1_Parity_c
#define gUART1_Parity_c                     gUartParityNone_c
#endif

#ifndef gUART1_DataBits_c
#define gUART1_DataBits_c                   gUart_8bits_c
#endif

#ifndef gUART1_Baudrate_c
#define gUART1_Baudrate_c                   gUARTBaudRate115200_c
#endif



#endif

/******************************************************************************
 * READ ONLY SECTION: THE USER SHALL NOT MODIFY THIS SECTION
 * Registers configuration is done by re-mapping the ports defined in specific
 * target header file 
 ******************************************************************************/
/********** 
 * UART_0 * 
 **********/
#if (gUart0_c == gUART_HW_MOD_0_c)
#define gUART_0_SIM_SCG_c                   gUART0_SIM_SCGC_REG_c           /* UART0 clock gating register */
#define gUART_0_SIM_SCG_BIT_c               gUART0_SIM_SCGC_BIT_c           /* UART0 clock gating bit */
#define gUART_0_PORT_SIM_SCG_c              gUART0_PORT_SIM_SCGC_REG_c      /* UART0 port clock gating register */
#define gUART_0_PORT_SIM_SGC_BIT_c          gUART0_PORT_SIM_SGC_BIT_c       /* UART0 port clock gating bit*/
#define gUART_0_RX_PCR_c                    gUART0_RX_PCR_REG_c             /* UART0 RX Pin Control register*/
#define gUART_0_TX_PCR_c                    gUART0_TX_PCR_REG_c             /* UART0 TX Pin Control register */
#define gUART_0_RTS_PCR_c                   gUART0_RTS_PCR_REG_c            /* UART0 RTS Pin Control register */
#define gUART_0_CTS_PCR_c                   gUART0_CTS_PCR_REG_c            /* UART0 CTS Pin Control register*/
#define gUART_0_MUX_c                       gUART0_PORT_MUX_c               /* UART0 RX, TX, RTS and CTS pins alternate function (see MUX) */
#define gUART_0_IRQ_NUM_c                   gUART0_IRQ_NUM_c                /* UART0 interrupt source number */
#define gUART_0_ERR_IRQ_NUM_c               gUART0_ERR_IRQ_NUM_c            /* UART0 error interrupt source number */
#elif (gUart1_c == gUART_HW_MOD_1_c)
#define gUART_0_SIM_SCG_c                   gUART1_SIM_SCGC_REG_c           /* UART1 clock gating register */
#define gUART_0_SIM_SCG_BIT_c               gUART1_SIM_SCGC_BIT_c           /* UART1 clock gating bit */
#define gUART_0_PORT_SIM_SCG_c              gUART1_PORT_SIM_SCGC_REG_c      /* UART1 port clock gating register */
#define gUART_0_PORT_SIM_SGC_BIT_c          gUART1_PORT_SIM_SGC_BIT_c       /* UART1 port clock gating bit*/
#define gUART_0_RX_PCR_c                    gUART1_RX_PCR_REG_c             /* UART1 RX Pin Control register*/
#define gUART_0_TX_PCR_c                    gUART1_TX_PCR_REG_c             /* UART1 TX Pin Control register */
#define gUART_0_RTS_PCR_c                   gUART1_RTS_PCR_REG_c            /* UART1 RTS Pin Control register */
#define gUART_0_CTS_PCR_c                   gUART1_CTS_PCR_REG_c            /* UART1 CTS Pin Control register*/
#define gUART_0_MUX_c                       gUART1_PORT_MUX_c               /* UART1 RX, TX, RTS and CTS pins alternate function (see MUX) */
#define gUART_0_IRQ_NUM_c                   gUART1_IRQ_NUM_c                /* UART1 interrupt source number */
#define gUART_0_ERR_IRQ_NUM_c               gUART1_ERR_IRQ_NUM_c            /* UART1 error interrupt source number */
#endif

  
  
/********** 
 * UART_1 * 
 **********/
#if (gUart1_c == gUART_HW_MOD_0_c)
#define gUART_1_SIM_SCG_c                   gUART0_SIM_SCGC_REG_c           /* UART0 clock gating register */
#define gUART_1_SIM_SCG_BIT_c               gUART0_SIM_SCGC_BIT_c           /* UART0 clock gating bit */
#define gUART_1_PORT_SIM_SCG_c              gUART0_PORT_SIM_SCGC_REG_c      /* UART0 port clock gating register */
#define gUART_1_PORT_SIM_SGC_BIT_c          gUART0_PORT_SIM_SGC_BIT_c       /* UART0 port clock gating bit*/
#define gUART_1_RX_PCR_c                    gUART0_RX_PCR_REG_c             /* UART0 RX Pin Control register*/
#define gUART_1_TX_PCR_c                    gUART0_TX_PCR_REG_c             /* UART0 TX Pin Control register */
#define gUART_1_RTS_PCR_c                   gUART0_RTS_PCR_REG_c            /* UART0 RTS Pin Control register */
#define gUART_1_CTS_PCR_c                   gUART0_CTS_PCR_REG_c            /* UART0 CTS Pin Control register*/
#define gUART_1_MUX_c                       gUART0_PORT_MUX_c               /* UART0 RX, TX, RTS and CTS pins alternate function (see MUX) */
#define gUART_1_IRQ_NUM_c                   gUART0_IRQ_NUM_c                /* UART0 interrupt source number */
#define gUART_1_ERR_IRQ_NUM_c               gUART0_ERR_IRQ_NUM_c            /* UART0 error interrupt source number */
#elif (gUart1_c == gUART_HW_MOD_1_c)
#define gUART_1_SIM_SCG_c                   gUART1_SIM_SCGC_REG_c           /* UART1 clock gating register */
#define gUART_1_SIM_SCG_BIT_c               gUART1_SIM_SCGC_BIT_c           /* UART1 clock gating bit */
#define gUART_1_PORT_SIM_SCG_c              gUART1_PORT_SIM_SCGC_REG_c      /* UART1 port clock gating register */
#define gUART_1_PORT_SIM_SGC_BIT_c          gUART1_PORT_SIM_SGC_BIT_c       /* UART1 port clock gating bit*/
#define gUART_1_RX_PCR_c                    gUART1_RX_PCR_REG_c             /* UART1 RX Pin Control register*/
#define gUART_1_TX_PCR_c                    gUART1_TX_PCR_REG_c             /* UART1 TX Pin Control register */
#define gUART_1_RTS_PCR_c                   gUART1_RTS_PCR_REG_c            /* UART1 RTS Pin Control register */
#define gUART_1_CTS_PCR_c                   gUART1_CTS_PCR_REG_c            /* UART1 CTS Pin Control register*/
#define gUART_1_MUX_c                       gUART1_PORT_MUX_c               /* UART1 RX, TX, RTS and CTS pins alternate function (see MUX) */
#define gUART_1_IRQ_NUM_c                   gUART1_IRQ_NUM_c                /* UART1 interrupt source number */
#define gUART_1_ERR_IRQ_NUM_c               gUART1_ERR_IRQ_NUM_c            /* UART1 error interrupt source number */
#endif




/******************************************************************************
 * Name: gUart0_InterruptPriority_c
 * Description: UART_0 IRQ priority 
 *****************************************************************************/

#define gUart0_InterruptPriority_c          (0x9u) 

/******************************************************************************
 * Name: gUart1_InterruptPriority_c
 * Description: UART_1 IRQ priority
 *****************************************************************************/

#define gUart1_InterruptPriority_c          (0x9u)

/******************************************************************************
 * Name: gUart0_ErrInterruptPriority_c
 * Description: UART_0 Error IRQ priority
 *****************************************************************************/

#define gUart0_ErrInterruptPriority_c       (0x8u) 

/******************************************************************************
 * Name: gUart1_ErrInterruptPriority_c
 * Description: UART_1 Error IRQ priority
 *****************************************************************************/

#define gUart1_ErrInterruptPriority_c       (0x8u)

/******************************************************************************
*******************************************************************************
* Public type definitions
*******************************************************************************
******************************************************************************/

/******************************************************************************
 * Name: UartBaudRate_t
 * Description: data type definition for UART baudrate
 *****************************************************************************/ 

typedef uint16_t UartBaudRate_t;

/******************************************************************************
 * Name: UartParityMode_t
 * Description: This data type enumerates the possible values of UART parity modes
 *****************************************************************************/ 

typedef enum UartParity_tag 
{
	gUartParityNone_c = 0,
	gUartParityEven_c,
	gUartParityOdd_c,
	gUartParityMax_c
} UartParityMode_t;

/******************************************************************************
 * Name: UartDataBits_t
 * Description: This data type enumerates the possible values of UART data bits 
 *****************************************************************************/
typedef enum UartDataBits_tag 
{
	gUart_8bits_c = 0,
	gUart_9bits_c,
	gUartDataBitsMax_c
} UartDataBits_t;

typedef enum UartHwFlowControl_tag
{
	gNoFlowControl_c = 0,
	gFlowEnabled_c
} UartHwFlowControl_t;

typedef enum UartRTSActiveState_tag
{
	gRtsActiveLow_c = 0,
	gRtsActiveHi_c
} UartRTSActiveState_t;

/******************************************************************************
*******************************************************************************
* Public prototypes
*******************************************************************************
******************************************************************************/

#if (gUart0_Enabled_d == FALSE) && (gUart1_Enabled_d == FALSE)

/******************************************************************************
 * Name: see section below
 * Description: stub functions if both UART ports are disabled 
 *****************************************************************************/

#define Uart_ModuleInit()
#define Uart_ModuleUninit()
#define Uart_ClearErrors()
#define Uart0_SetBaud(baudRate)
#define Uart1_SetBaud(baudRate)
#define Uart0_Transmit(pBuf, bufLen, pfCallBack)    FALSE
#define Uart1_Transmit(pBuf, bufLen, pfCallBack)    FALSE
#define Uart0_IsTxActive()                          FALSE
#define Uart1_IsTxActive()                          FALSE
#define Uart0_SetRxCallBack(pfCallBack)
#define Uart1_SetRxCallBack(pfCallBack)
#define Uart0_GetByteFromRxBuffer(pDst)             FALSE
#define Uart1_GetByteFromRxBuffer(pDst)             FALSE
#define Uart0_UngetByte(byte)
#define Uart1_UngetByte(byte)
#define Uart0_RxBufferByteCount                     0
#define Uart1_RxBufferByteCount                     0

#else /* (gUart0_Enabled_d == FALSE) && (gUart1_Enabled_d == FALSE) */

/******************************************************************************
* Uart_ModuleInit initializes the serial port(s) and module internal variables
*     
* Interface assumptions:
*   It should be called before any the other driver function. It needs to be
*   called first before use UART driver.
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/

extern void Uart_ModuleInit
(
    void
);

/******************************************************************************
* Uart_ModuleUnInit shut down the UART module by clearing the SCI Control 
* registers. 
*     
*
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart_ModuleUninit
(
    void
);

/******************************************************************************
* Uart_ClearErrors clears any error flags seen by the UART driver. In rare 
* conditions, an error flag can be set without triggering an error interrupt, 
* and will prevent Rx and/or Tx interrupts from occurring. The most likely cause 
* is a breakpoint set during debugging, when a UART port is active.     
* Calling this function occasionally allows the UART code to recover    
* from these errors. 
*     
*
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart_ClearErrors
(
    void
);

/******************************************************************************
* Uart0_GetByteFromRxBuffer Retrieve one byte from 
* the driver's Rx buffer and store it at *pDst.
*     
* Interface assumptions:
*   NONE
*
* return value:
*   TRUE -  If a byte was retrieved
*   FALSE - If the Rx buffer is empty.
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern bool_t Uart0_GetByteFromRxBuffer
(
    unsigned char *pDst
);

/******************************************************************************
* Uart1_GetByteFromRxBuffer Retrieve one byte from 
* the driver's Rx buffer and store it at *pDst.
*     
* Interface assumptions:
*   NONE
*
* return value:
*   TRUE -  If a byte was retrieved
*   FALSE - If the Rx buffer is empty.
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern bool_t Uart1_GetByteFromRxBuffer
(
    unsigned char *pDst
);

/******************************************************************************
* Uart0_IsTxActive checks if a transmission is active.
*     
*
* Interface assumptions:
*   NONE
*
* return value:
*   TRUE - If there are any bytes in the transmit buffers that have not be sent.
*   FALSE - If there are no bytes to be sent.
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern bool_t Uart0_IsTxActive
(
    void
);

/******************************************************************************
* Uart1_IsTxActive checks if a transmission is active.
*     
*
* Interface assumptions:
*   NONE
*
* return value:
*   TRUE - If there are any bytes in the transmit buffers that have not be sent.
*   FALSE - If there are no bytes to be sent.
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern bool_t Uart1_IsTxActive
(
    void
);

/******************************************************************************
* Uart1_SetRxCallBack sets the receive side callback function. 
* This function will be called whenever a byte is received by the driver. 
* The callback function will be called in interrupt context, so it should be 
* kept very short. If the callback pointer is set to NULL, the Rx interrupt 
* will be turned off, and all data in the driver's Rx buffer will be discarded. 
*
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart0_SetRxCallBack
(
    void (*pfCallBack)(void)  /*IN: Pointer to that callback function to be executed 
                              when any of the receiving conditions is reached*/
);

/******************************************************************************
* Uart2_SetRxCallBack sets the receive side callback function. 
* This function will be called whenever a byte is received by the driver. 
* The callback function will be called in interrupt context, so it should be 
* kept very short. If the callback pointer is set to NULL, the Rx interrupt 
* will be turned off, and all data in the driver's Rx buffer will be discarded. 
*
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart1_SetRxCallBack
(
    void (*pfCallBack)(void)  /*IN: Pointer to that callback function to be executed 
                              when any of the receiving conditions is reached*/
);

/******************************************************************************
* Uart0_SetBaud set the selected baud rate specified in the given parameter.
*     
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart0_SetBaud
(
    uint32_t baudRate    /* IN: Selected baud rate */
);

/******************************************************************************
* Uart1_SetBaud set the selected baud rate specified in the given parameter.
*     
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart1_SetBaud
(
	uint32_t baudRate   /* IN: Selected baud rate */
);

/******************************************************************************
* Uart0_Transmit Transmits bufLen bytes of data from pBuffer 
* over a port. Call *pfCallBack() when the entire buffer has been sent.  
* The caller must ensure that the buffer remains available until the call back 
* function is called. pfCallBack must not be NULL.  This function is executed 
* as part of an interrupt service routine, as a result follow all the 
* recommendations for interrupt execution timing and context
*
* Interface assumptions:
*   NONE
*
* return value:
*   FALSE - If there are no more available Tx buffer slots.
*   TRUE - If there are slots to transmit.
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern bool_t Uart0_Transmit
(
    //unsigned char const *pBuf,                      /* IN: Pointer to the buffer that is required to be 
    unsigned char  *pBuf,                      /* IN: Pointer to the buffer that is required to be 
                                                    transmitted */
    index_t bufLen,                                 /* IN: Number of bytes to be transmitted */
    //void (*pfCallBack)(unsigned char const *pBuf)   /*IN: Pointer to that callback function to be executed 
    void (*pfCallBack)(unsigned char *pBuf)   /*IN: Pointer to that callback function to be executed 
					            when the entire buffer has been sent*/
);

/******************************************************************************
* Uart1_Transmit Transmits bufLen bytes of data from pBuffer 
* over a port. Call *pfCallBack() when the entire buffer has been sent.  
* The caller must ensure that the buffer remains available until the call back 
* function is called. pfCallBack must not be NULL.  This function is executed 
* as part of an interrupt service routine, as a result follow all the 
* recommendations for interrupt execution timing and context
*
* Interface assumptions:
*   NONE
*
* return value:
*   FALSE - If there are no more available Tx buffer slots.
*   TRUE - If there are slots to transmit.
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern bool_t Uart1_Transmit
(
    //unsigned char const *pBuf,                      /* IN: Pointer to the buffer that is required to be 
    unsigned char *pBuf,                      /* IN: Pointer to the buffer that is required to be 
                                                    transmitted */
    index_t bufLen,                                 /* IN: Number of bytes to be transmitted */
    //void (*pfCallBack)(unsigned char const *pBuf)   /*IN: Pointer to that callback function to be executed 
    void (*pfCallBack)(unsigned char *pBuf)   /*IN: Pointer to that callback function to be executed 
					            when the entire buffer has been sent*/
);                                

/******************************************************************************
* Uart0_UngetByte If a caller reads  many 
* characters, it can unget it. Calling the unget routine will store 
* the byte to be read by the next call to Uart1_GetByteFromRxBuffer(), 
* and will also call the Rx call back. 
*     
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart0_UngetByte
(
    unsigned char byte    /* IN: Unget byte to be read by 
		          the Uart1_GetByteFromRxBuffer() in its next call*/
);

/******************************************************************************
* Uart1_UngetByte If a caller reads  many 
* characters, it can unget it. Calling the unget routine will store 
* the byte to be read by the next call to Uart1_GetByteFromRxBuffer(), 
* and will also call the Rx call back. 
*     
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart1_UngetByte
(
    unsigned char byte    /* IN: Unget byte to be read by 
			  the Uart2_GetByteFromRxBuffer() in its next call*/
);

/******************************************************************************
* Uart0_RxBufferByteCount returns unread bytes number from UART_1 buffer 
*     
* Interface assumptions:
*   NONE
*
* return value:
*   uint32_t - the number of unread bytes
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern uint32_t Uart0_RxBufferByteCount
(
    void
); 

/******************************************************************************
* Uart1_RxBufferByteCount returns unread bytes number from UART_1 buffer 
*     
* Interface assumptions:
*   NONE
*
* return value:
*   uint32_t - the number of unread bytes
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern uint32_t Uart1_RxBufferByteCount
(
    void
);


/******************************************************************************
* Uart1Isr is the UART_0 Interrupt Service Routine
*     
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart0Isr
(
    void
);

/******************************************************************************
* Uart2Isr is the UART_1 Interrupt Service Routine
*     
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart1Isr
(
    void
);

/******************************************************************************
* Uart0ErrorIsr clears enabled UART_1 errors
*     
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart0ErrorIsr
(
    void
); 

/******************************************************************************
* Uart1ErrorIsr clears enabled UART_1 errors
*     
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart1ErrorIsr
(
    void
);


/******************************************************************************
* Uart0InterruptHandler checks where the interrupt comes
*     
* Interface assumptions:
*   NONE
*
* return value:
*   NONE
*   
* Effects on global data:
*   NONE
*
*******************************************************************************/
extern void Uart0InterruptHandler
(
    void
);

/*---------------------------------------------------------------------------
 * Name: Uart_ModuleEnableLowPowerWakeup
 * Description: Enables the Receive input active edge, RXEDGIF, to generate interrupt requests.
 *              The receive input active edge detect circuit is still active on low power modes
 *              (wait and stop). An active edge on the receive input brings the CPU out of low 
 *              power mode if the interrupt is not masked.
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
extern void Uart_ModuleEnableLowPowerWakeup
(
    void
);

/*---------------------------------------------------------------------------
 * Name: Uart_ModuleDisableLowPowerWakeup
 * Description: Disables the Receive input active edge, RXEDGIF, to generate interrupt requests.
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
extern void Uart_ModuleDisableLowPowerWakeup
(
    void
);

/*---------------------------------------------------------------------------
 * Name: Uart_IsWakeUpSource
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
extern bool_t Uart_IsWakeUpSource
(
    void
);

/****************************************************************************** 
 * Declare the default API. 
 * If gUart_PortDefault_d is not defined, there is 
 * no default API. 
 *****************************************************************************/
#if gUart_PortDefault_d == 0
#define UartX_Transmit              Uart0_Transmit  
#define UartX_IsTxActive            Uart0_IsTxActive
#define UartX_SetRxCallBack         Uart0_SetRxCallBack
#define UartX_GetByteFromRxBuffer   Uart0_GetByteFromRxBuffer
#define UartX_SetBaud               Uart0_SetBaud
#define UartX_UngetByte             Uart0_UngetByte
#define UartX_TxCompleteFlag        Uart0_TxCompleteFlag
#define UartX_RxBufferByteCount     Uart0_RxBufferByteCount
#define UartX_DefaultBaud           gUART0_Baudrate_c
#endif

#if gUart_PortDefault_d == 1
#define UartX_Transmit              Uart1_Transmit  
#define UartX_IsTxActive            Uart1_IsTxActive
#define UartX_SetRxCallBack         Uart1_SetRxCallBack
#define UartX_GetByteFromRxBuffer   Uart1_GetByteFromRxBuffer
#define UartX_SetBaud               Uart1_SetBaud
#define UartX_UngetByte             Uart1_UngetByte
#define UartX_TxCompleteFlag        Uart1_TxCompleteFlag
#define UartX_RxBufferByteCount     Uart1_RxBufferByteCount
#define UartX_DefaultBaud           gUART1_Baudrate_c
#endif


#endif /* (gUart0_Enabled_d == FALSE) && (gUart1_Enabled_d == FALSE) */
  
#endif /* UART_INTERFACE_H_ */
